<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");
header("Content-Type: application/json");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit();
}

require_once 'connect.php';

// Check if this is a batch request
$isBatchRequest = isset($_GET['accountNumbers']);

try {
    if ($isBatchRequest) {
        // Handle batch request
        $accountNumbersJson = $_GET['accountNumbers'] ?? '[]';
        $accountNumbers = json_decode($accountNumbersJson, true);
        
        if (!is_array($accountNumbers) || empty($accountNumbers)) {
            http_response_code(400);
            echo json_encode([
                "status" => "error",
                "code" => 400,
                "message" => "Valid accountNumbers array parameter is required",
                "data" => null
            ]);
            exit;
        }
        
        // Validate and sanitize each account number
        $validAccountNumbers = [];
        foreach ($accountNumbers as $accountNumber) {
            $accountNumber = trim($accountNumber);
            if (preg_match('/^[a-zA-Z0-9\-]+$/', $accountNumber)) {
                $validAccountNumbers[] = $accountNumber;
            }
        }
        
        if (empty($validAccountNumbers)) {
            http_response_code(400);
            echo json_encode([
                "status" => "error",
                "code" => 400,
                "message" => "No valid account numbers provided",
                "data" => null
            ]);
            exit;
        }
        
        // Prepare the query with placeholders
        $placeholders = implode(',', array_fill(0, count($validAccountNumbers), '?'));
        $types = str_repeat('s', count($validAccountNumbers));
        
        $stmt = $conn->prepare("
            SELECT * FROM uploaded_survey_data 
            WHERE accountNumber IN ($placeholders)
            ORDER BY dateCaptured DESC
        ");
        
        if (!$stmt) {
            throw new Exception("Database prepare statement failed: " . $conn->error);
        }
        
        // Bind parameters dynamically
        $stmt->bind_param($types, ...$validAccountNumbers);
        
        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }
        
        $result = $stmt->get_result();
        $allSubmissions = $result->fetch_all(MYSQLI_ASSOC);
        
        // Organize data by account number
        $groupedData = [];
        foreach ($allSubmissions as $submission) {
            $accountNo = $submission['accountNumber'];
            if (!isset($groupedData[$accountNo])) {
                $groupedData[$accountNo] = [];
            }
            $groupedData[$accountNo][] = $submission;
        }
        
        // Ensure all requested accounts are in the response, even if empty
        $responseData = [];
        foreach ($validAccountNumbers as $accountNo) {
            $responseData[$accountNo] = $groupedData[$accountNo] ?? [];
        }
        
        http_response_code(200);
        echo json_encode([
            "status" => "success",
            "code" => 200,
            "message" => "Batch data retrieved successfully",
            "data" => $responseData
        ]);
        
    } else {
        // Handle single account request (original functionality)
        $accountNumber = $_GET['accountNumber'] ?? null;
        
        if (empty($accountNumber)) {
            http_response_code(400);
            echo json_encode([
                "status" => "error",
                "code" => 400,
                "message" => "Valid accountNumber parameter is required",
                "data" => null
            ]);
            exit;
        }
        
        $accountNumber = trim($accountNumber);
        if (!preg_match('/^[a-zA-Z0-9\-]+$/', $accountNumber)) {
            http_response_code(400);
            echo json_encode([
                "status" => "error",
                "code" => 400,
                "message" => "Invalid account number format",
                "data" => null
            ]);
            exit;
        }
        
        $stmt = $conn->prepare("
            SELECT * FROM uploaded_survey_data 
            WHERE accountNumber = ?
            ORDER BY dateCaptured DESC
            LIMIT 1
        ");
        
        if (!$stmt) {
            throw new Exception("Database prepare statement failed: " . $conn->error);
        }
        
        $stmt->bind_param("s", $accountNumber);
        
        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }
        
        $result = $stmt->get_result();
        $submissions = $result->fetch_all(MYSQLI_ASSOC);
        
        if (empty($submissions)) {
            http_response_code(404);
            echo json_encode([
                "status" => "error",
                "code" => 404,
                "message" => "No survey data found for account: $accountNumber",
                "data" => null
            ]);
        } else {
            http_response_code(200);
            echo json_encode([
                "status" => "success",
                "code" => 200,
                "message" => "Data retrieved successfully",
                "data" => $submissions
            ]);
        }
    }
} catch (Exception $e) {
    error_log("API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        "status" => "error",
        "code" => 500,
        "message" => "Internal server error: " . $e->getMessage(),
        "data" => null
    ]);
} finally {
    if (isset($stmt)) $stmt->close();
    $conn->close();
}
?>