<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");
header("Content-Type: application/json");

error_reporting(E_ALL);
ini_set('display_errors', 1);

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

include "connect.php";

if ($conn->connect_error) {
    echo json_encode([
        "success" => false,
        "message" => "Database connection failed: " . $conn->connect_error
    ]);
    exit();
}

$response = [
    "success" => false,
    "message" => "",
    "insertedCount" => 0,
    "duplicateCount" => 0,
    "errors" => [],
    "duplicates" => []
];

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    $response["message"] = "Invalid request method";
    echo json_encode($response);
    exit();
}

if (!isset($_FILES['csvFile']) || $_FILES['csvFile']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(400);
    $response["message"] = "No file uploaded or upload error";
    echo json_encode($response);
    exit();
}

$csvFile = $_FILES['csvFile']['tmp_name'];
$handle = fopen($csvFile, 'r');
if (!$handle) {
    http_response_code(400);
    $response["message"] = "Could not open CSV file";
    echo json_encode($response);
    exit();
}

$header = fgetcsv($handle);
if (!$header) {
    http_response_code(400);
    $response["message"] = "Could not read CSV headers";
    echo json_encode($response);
    exit();
}

$conn->begin_transaction();
$insertedCount = 0;
$duplicateCount = 0;

try {
    while (($row = fgetcsv($handle)) !== false) {
        if (empty($row) || count($row) !== count($header)) {
            $response["errors"][] = "Skipped row due to column count mismatch";
            continue;
        }

        $rowData = array_combine($header, $row);
        if (!isset($rowData['accountNumber'])) {
            $response["errors"][] = "Skipped row with missing accountNumber";
            continue;
        }

        $accountNumber = $rowData['accountNumber'];

        $checkStmt = $conn->prepare("SELECT 1 FROM uploaded_survey_data WHERE accountNumber = ?");
        $checkStmt->bind_param("s", $accountNumber);
        $checkStmt->execute();
        $exists = $checkStmt->get_result()->fetch_row();
        $checkStmt->close();
        
        if ($exists) {
            $duplicateCount++;
            $response["duplicates"][] = $accountNumber;
            continue; 
        }

        $columns = array_keys($rowData);
        $placeholders = implode(',', array_fill(0, count($columns), '?'));
        $columnList = '`' . implode('`,`', $columns) . '`';

        $types = '';
        $values = [];
        foreach ($rowData as $value) {
            if (is_numeric($value)) {
                $types .= (strpos($value, '.') !== false) ? 'd' : 'i';
                $values[] = (strpos($value, '.') !== false) ? (float)$value : (int)$value;
            } else {
                $types .= 's';
                $values[] = $value;
            }
        }

        $stmt = $conn->prepare("INSERT INTO uploaded_survey_data ($columnList) VALUES ($placeholders)");
        
        if (!$stmt) {
            $response["errors"][] = "Prepare failed: " . $conn->error;
            continue;
        }

        if (!$stmt->bind_param($types, ...$values)) {
            $response["errors"][] = "Bind failed: " . $stmt->error;
            $stmt->close();
            continue;
        }

        if (!$stmt->execute()) {
            $response["errors"][] = "Insert error: " . $stmt->error;
            $stmt->close();
            continue;
        }

        $insertedCount++;
        $stmt->close();
    }

    $conn->commit();
    
    $response["success"] = true;
    $response["insertedCount"] = $insertedCount;
    $response["duplicateCount"] = $duplicateCount;
    $response["message"] = sprintf(
        "Processed CSV. Inserted %d rows, skipped %d duplicates.",
        $insertedCount,
        $duplicateCount
    );

} catch (Exception $e) {
    $conn->rollback();
    $response["message"] = "Transaction failed: " . $e->getMessage();
    http_response_code(500);
} finally {
    fclose($handle);
}

echo json_encode($response);
$conn->close();
?>